import json
from datetime import datetime
from dateutil import tz
import boto3
import urllib.request

# Initialize boto3 client for Timestream queries
client = boto3.client('timestream-query')


def get_timestream_data(from_date, to_date):
    # SQL query to get average power consumption per day
    query = '''SELECT 
    DATE_TRUNC('day', time) AS day,
    AVG(
        measure_value::bigint
    ) AS avg_power_consumption
FROM "smart_meter_database"."smart_meter_table" 
WHERE time BETWEEN 'from_date' AND 'to_date'
GROUP BY DATE_TRUNC('day', time)
ORDER BY day'''
    query = query.replace("from_date", from_date).replace("to_date", to_date)
    print(query)
    return client.query(QueryString=query)

    
def get_current_date():
    jst_tz = tz.gettz('Asia/Tokyo')

    # Get timezone for Japan
    utc_now = datetime.utcnow()
    jst_now = utc_now.replace(tzinfo=tz.tzutc()).astimezone(jst_tz)

    # Format date as yyyymmdd
    formatted_date = jst_now.strftime("%Y%m%d")

    return formatted_date


def lambda_handler(event, context):
    
    # Extract relevant information from the event
    agent = event['agent']
    actionGroup = event['actionGroup']
    function = event['function']
    parameters = event.get('parameters', [])
    
    # Handle different function calls
    if function == "getCurrentDate":
        response = get_current_date()
    elif function == "getdayPowerConsumption":
        from_date = next(p for p in parameters if p["name"] == "from_date")["value"]
        to_date = next(p for p in parameters if p["name"] == "to_date")["value"]
        response = json.dumps(get_timestream_data(from_date, to_date))
    else:
        response = ""

    # Prepare the response body
    responseBody =  {
        "TEXT": {
            "body": response
        }
    }
    
    # Construct the action response
    action_response = {
        'actionGroup': actionGroup,
        'function': function,
        'functionResponse': {
            'responseBody': responseBody
        }
    }
    
    # Prepare the final response
    lambda_function_response = {'response': action_response, 'messageVersion': event['messageVersion']}
    print("Response: {}".format(lambda_function_response))
    return lambda_function_response