import sys
import serial
import time
import os
import json
import glob
import math
import traceback
import serial.tools.list_ports

import awsiot.greengrasscoreipc  #import whole awsiot
from awsiot.greengrasscoreipc.clientv2 import GreengrassCoreIPCClientV2
from awsiot.greengrasscoreipc.model import (
    PublishToIoTCoreRequest,
    GetSecretValueRequest,
    GetSecretValueResponse,
    ServiceError,
    QOS,
)

# Instantiate Greengrass IPC client
ipc_client = GreengrassCoreIPCClientV2()

# こちらのWebサイトを参考にさせていただきました。https://www.ratoc-e2estore.com/blog/2023/06/wsuha-02
# Econet Lite DATA frame constants
ehd1 = 0x10
ehd2 = 0x81
tidh = 0x52
tidl = 0x53
Seoj_x1 = 0x05
Seoj_x2 = 0xFF
Seoj_x3 = 0x01
Deoj_x1 = 0x02
Deoj_x2 = 0x88
Deoj_x3 = 0x01
esv_req = 0x62
opc_req = 0x01
epc_req = 0xE7
pdc_req = 0x00

# Request command
req_cmd = bytes([ehd1, ehd2, tidh, tidl, Seoj_x1, Seoj_x2, Seoj_x3, Deoj_x1, Deoj_x2, Deoj_x3, esv_req, opc_req, epc_req, pdc_req])


# AWS IoT Core settings
AWS_IOT_TOPIC = "power/consumption"

# Recipe Configuration variables
UPDATE_INTERVAL = 5 # Default value
SECRET_NAME = "RouteB_Secret"   # Default value
SERIAL_PORT = "/dev/ttyUSB0"   # Default value
SIMULATION_MODE = False # Simulation mode flag


def get_configuration():
    """Function to retrieve Greengrass Component configuration"""
    global UPDATE_INTERVAL, SECRET_NAME, SIMULATION_MODE,  SECRET_NAME
    try:
        config = ipc_client.get_configuration()
        if hasattr(config, 'value'):
            config_dict = config.value
            UPDATE_INTERVAL = int(config_dict.get('UPDATE_INTERVAL', 5))

            if UPDATE_INTERVAL < 1:
                print(f"UPDATE_INTERVAL {UPDATE_INTERVAL} is too low.  Setting it to 1。")
                UPDATE_INTERVAL = 1


            SECRET_NAME = config_dict.get('SECRET_NAME', "RouteB_Secret")
            SERIAL_PORT = config_dict.get('SERIAL_PORT', "/dev/ttyUSB0")
            SIMULATION_MODE = config.get('SIMULATION_MODE', "False").lower() == "true"
        else:
            print("Configuration doesn't have 'value' attribute. Using default values.")
    except Exception as e:
        print(f"Error getting configuration: {e}. Using default values.")
    print(f"UPDATE_INTERVAL set to: {UPDATE_INTERVAL}")
    print(f"SECRET_NAME set to: {SECRET_NAME}")
    print(f"SERIAL_PORT set to: {SERIAL_PORT}")
    print(f"SIMULATION_MODE set to: {SIMULATION_MODE}")


def get_secret():
    """Function to retrieve secret from AWS Secrets Manager"""
    TIMEOUT = 10
    try:

        # Get secret (asynchronous operation)
        future_response = ipc_client.get_secret_value_async(secret_id=SECRET_NAME, version_stage='AWSCURRENT')

        # Get response
        response = future_response.result(TIMEOUT)

        # Convert secret result to JSON format
        secret_json = json.loads(response.secret_value.secret_string)

        print("Retrieved secret from Secrets Manager")
        return secret_json['rbpwd'], secret_json['rbid']

    except ServiceError as e:
        # Output ServiceError details
        print(f"ServiceError occurred: {str(e)}")
        if hasattr(e, 'message'):
            print(f"ServiceError message: {e.message}")
        if hasattr(e, 'context'):
            print(f"ServiceError context: {e.context}")
        print(f"Error type: {type(e)}")
        print(f"Function name: {get_secret.__name__}")
        traceback.print_exc()
        raise

    except Exception as e:
        # Output other error details
        print(f"Error retrieving secret: {e}")
        print(f"Error type: {type(e)}")
        print(f"Function name: {get_secret.__name__}")
        traceback.print_exc()
        raise

# Get configuration
try:
    get_configuration()
except Exception as e:
    print(f"Error in get_configuration: {e}")
    print(f"Error type: {type(e)}")
    import traceback
    traceback.print_exc()

def simulate_power_data():
    """Generate simulated power data"""
    current_time = time.time()
    return int(250 + 50 * math.sin(current_time / 10))

def receive_msg():
    lpcont = 1
    while lpcont:
        rcvmsg = ser.readline()
        print("Received message:", rcvmsg)
        if rcvmsg.startswith(b"ERXUDP"):
            rcv_txt = rcvmsg.strip().split(b' ')
            rcv_edata = rcv_txt[9]
            print("EDATA;", rcv_edata, "\r\n")
            rcv_tid = rcv_edata[4:4 + 4]
            print("Received TID:", rcv_tid)
            if rcv_tid == b'5253':
                lpcont = 0
            else:
                lpcont = 1
        else:
            lpcont = 0
    return rcvmsg

def receive_echoback(skcmd):
    lpcont = 1
    while lpcont:
        rcv_echobk = ser.readline()
        if rcv_echobk == b"":
            lpcont = 2
            return lpcont
        if rcv_echobk.startswith(skcmd):
            print("Echoback : ", rcv_echobk)
            lpcont = 0
        else:
            lpcont = 1
    return lpcont

def receive_OK():
    lpcont = 1
    while lpcont:
        rcvmsg = ser.readline()
        if rcvmsg == b'':
            lpcont = 2
            return lpcont
        if rcvmsg.startswith(b"OK"):
            print("Ack : ", rcvmsg, "\r\n")
            lpcont = 0
        else:
            lpcont = 1
    return rcvmsg

def send_data_to_aws(power_data, timestamp):
    try:
        ipc_client.publish_to_iot_core(
            topic_name=AWS_IOT_TOPIC,
            qos=QOS.AT_LEAST_ONCE,
            payload=json.dumps({
                "power_consumption": power_data,
                "timestamp": timestamp,
                "UPDATE_INTERVAL": UPDATE_INTERVAL
            }).encode()
        )
        print("Message published successfully")
    except Exception as e:
        print(f"Failed to publish message. Error: {str(e)}")
        print(f"Error Type: {type(e).__name__}")
        print(f"Error Details: {traceback.format_exc()}")


def get_edata(ipv6Addr, req_cmd):
    if SIMULATION_MODE:
        return simulate_power_data()
    
    print("Send Request Command ")
    ser.write(b"SKSENDTO 1 " + ipv6Addr + b" 0E1A 1 0 000E ")
    ser.write(req_cmd)

    skcmd = b"SKSENDTO"
    rtn_cd = receive_echoback(skcmd)
    rtn_cd = receive_OK()

    rcvmsg = receive_msg()
    print(rcvmsg, "\r\n")

    if rcvmsg.startswith(b"ERXUDP"):
        print("ERXUDP received ", "\r\n")
        cols = rcvmsg.strip().split(b' ')
        res = cols[9]
        print("EDATA;", res)
        seoj = res[8:8 + 6]
        print("SEOJ:", seoj)
        ESV = res[20:20 + 2]
        print("ESV:", ESV)
        if seoj == b'028801' and ESV == b'72':
            EPC = res[24:24 + 2]
            print("EPC:", EPC, "\r\n")
            if EPC == b'E7':
                hexPower = res[28:28 + 8]
                print("Value:", hexPower, "\r\n")
                intpower = int(hexPower, 16)
                print(f"Power consumption: {intpower}")
                return intpower
    
    print("Failed to get valid power data")
    return None  # Return None in case of error

def main_loop():
    global prv_pwr
    global ipv6Addr
    global req_cmd
    
    while True:
        current_time = time.time()
        if not hasattr(main_loop, "last_update"):
            main_loop.last_update = current_time
        
        if current_time - main_loop.last_update >= UPDATE_INTERVAL:
            pwr_new = get_edata(ipv6Addr, req_cmd)
            now = time.strftime("%H:%M:%S")
            
            if pwr_new is not None:
                prv_pwr = pwr_new
                main_loop.last_update = current_time
                print(f"Updated power consumption: {pwr_new} at {now}")
                send_data_to_aws(pwr_new, now)
            else:
                print(f"Failed to get valid power data at {now}")
        
        time.sleep(1)

if not SIMULATION_MODE:
    # Serial port settings
    bitRate = 115200
    CRLF = b'\r\n'

    try:
        rbpwd, rbid = get_secret()
        print("Successfully retrieved secrets from Secrets Manager")
    except serial.SerialException as e:
        print(f"Failed to open serial port {SERIAL_PORT}: {e}")
        sys.exit(1)

    print(f"\r\nWSUHA & SmartMeter communication test 6R0 on {SERIAL_PORT}\r\n")


    ser = serial.Serial(SERIAL_PORT, bitRate, timeout=10)


    # Reset WSUHA command buffer
    ser.write(b'SKRESET\r\n')
    skcmd = b"SKRESET"
    rtn_cd = receive_echoback(skcmd)
    rtn_cd = receive_OK()

    # Set B-route Authentication Password
    ser.write(b'SKSETPWD C ')
    ser.write(rbpwd.encode('utf-8'))
    ser.write(CRLF)
    skcmd = b"SKSETPWD"
    rtn_cd = receive_echoback(skcmd)
    rtn_cd = receive_OK()

    # Set B-route Authentication ID
    ser.write(b'SKSETRBID ')
    ser.write(rbid.encode('utf-8'))
    ser.write(CRLF)
    skcmd = b"SKSETRBID"
    rtn_cd = receive_echoback(skcmd)
    rtn_cd = receive_OK()

    #  B-Route scan to find SmartMeter
    scanduration = 6
    scan_data = {}

    print("\r\nScan start to detect SmartMeter\r\n")
    ser.write(b"SKSCAN 2 FFFFFFFF 6 0 \r\n")
    skcmd = b"SKSCAN"
    rtn_cd = receive_echoback(skcmd)
    rtn_cd = receive_OK()

    # Retry SCAN to get some responce from SmartMeter
    scan_loop = True
    while scan_loop:
        rcvmsg = receive_msg()
        print("Scan Result:", rcvmsg, "\r\n", end="")

        if rcvmsg.startswith(b"EVENT 22"):
        # SCAN ends : receive EVENT 22
            scan_loop = False
        elif rcvmsg.startswith(b"  "):
            cols = rcvmsg.strip().split(b':')
            scan_data[cols[0]] = cols[1]

    # pull out Channel and set it to S2 reg.
    ser.write(b"SKSREG S2 " + scan_data[b"Channel"] + b"\r\n")
    skcmd = b"SKSREG"
    rtn_cd = receive_echoback(skcmd)
    rtn_cd = receive_OK()

    # pull out Pan ID and set it to S3 reg.
    ser.write(b"SKSREG S3 " + scan_data[b"Pan ID"] + b"\r\n")
    skcmd = b"SKSREG"
    rtn_cd = receive_echoback(skcmd)
    rtn_cd = receive_OK()

    # Convert MAC Address(64bit) to IPV6 address
    ser.write(b"SKLL64 " + scan_data[b"Addr"] + b"\r\n")
    skcmd = b"SKLL64"
    rtn_cd = receive_echoback(skcmd)
    ipv6Addr = ser.readline().strip()
    print("IPv6 Address:", ipv6Addr, "\r\n")

    # start to set up PANA Connection sequence
    ser.write(b"SKJOIN " + ipv6Addr + b"\r\n")
    skcmd = b"SKJOIN"
    rtn_cd = receive_echoback(skcmd)
    rtn_cd = receive_OK()

    # Wait PANA Connection completes until receive EVENT25.
    bConnected = False
    while not bConnected:
        rcvmsg = receive_msg()
        print(rcvmsg, "\r\n")
        if rcvmsg.startswith(b"EVENT 24"):
            print("PANA Connection failed.", "\r\n")
            sys.exit()
        elif rcvmsg.startswith(b"EVENT 25"):
            print("PANA Connection completed.", "\r\n")
            bConnected = True

    ser.timeout = 3
else:
    print("Running in simulation mode. No actual connection will be made.")
    ipv6Addr = b"SIMULATED_ADDRESS"

main_loop()