import json
import boto3
import base64
import logging

logger = logging.getLogger()
logger.setLevel(logging.INFO)

s3_client = boto3.client('s3')
iot_data_client = boto3.client('iot-data')
bedrock_client = boto3.client('bedrock-runtime')

def lambda_handler(event, context):
    bucket_name = event['Records'][0]['s3']['bucket']['name']
    key_name = event['Records'][0]['s3']['object']['key']
    
    response = s3_client.get_object(Bucket=bucket_name, Key=key_name)
    file_content = response['Body'].read()
    
    # Base64 encoding
    image_b64_string = base64.b64encode(file_content).decode("utf-8")
    
    prompt = """
    ポーカーについて質問します。この画像から各カードを読み取り、順番を考慮して、１行目は以下の例のフォーマットで教えてください。

    {"hand": ["♥10","♦J","♣3","♠A","♠A"], "hand_name": "ワンペア"}

    次に、２行目以降でこのポーカーハンドの場合、何をドローすればいいかアドバイスしてください。その際、以下を参考にしてください。

    ロイヤルフラッシュ（Royal Flush）は同じスートの10、ジャック、クイーン、キング、エースがある
    ストレートフラッシュ（Straight Flush）は同じスートでランクが連続する5枚のカードがある
    フォーカード（Four of a Kind）は同じランクのカードが4枚ある
    フルハウス（Full House）は3枚の同じランクと2枚の同じランクのカードがある
    フラッシュ（Flush）は同じスートの任意の5枚のカードがある
    ストレート（Straight）は異なるスートでランクが連続する5枚のカードがある
    スリーカード（Three of a Kind）は同じランクのカードが3枚ある
    ツーペア（Two Pair）は同じランクのカードのペアが2つある。たとえば["♣2", "♥2", "♠A", "♣J", "♠J"]はツーペアです。
    ワンペア（One Pair）は同じランクのカードのペアが1つある
    ハイカード（High Card）は上記のどの役も成立しない場合。その場合は、最も高いカードのランク入っているほうがよい。
    """
    
    request_body = json.dumps({
        "anthropic_version": "bedrock-2023-05-31",
        "max_tokens": 1024,
        "messages": [{
            "role": "user",
            "content": [
                {
                    "type": "image",
                    "source": {
                        "type": "base64",
                        "media_type": "image/jpeg", 
                        "data": image_b64_string
                    }
                },
                {
                    "type": "text",
                    "text": prompt
                }
            ]
        }]
    })
    
    try:
        response = bedrock_client.invoke_model(
            modelId="anthropic.claude-3-sonnet-20240229-v1:0",
            body=request_body
        )
        
        result = json.loads(response['body'].read())
        output_list = result.get("content", [])
        
        for output in output_list:
            message = output["text"]
            logger.info(message)
            
            # Publish to IoT Core
            iot_data_client.publish(
                topic='poker-advice',
                qos=1,
                payload=json.dumps({"message": message})
            )
        
        return {
            'statusCode': 200,
            'body': json.dumps('Success')
        }
        
    except Exception as e:
        logger.error(f"Error: {str(e)}")
        return {
            'statusCode': 500,
            'body': json.dumps('Error')
        }


