import * as cdk from 'aws-cdk-lib';
import { Construct } from 'constructs';
import * as s3 from 'aws-cdk-lib/aws-s3';
import * as iam from 'aws-cdk-lib/aws-iam';
import * as lambda from 'aws-cdk-lib/aws-lambda';
import * as s3n from 'aws-cdk-lib/aws-s3-notifications';


export class CdkStack extends cdk.Stack {
  constructor(scope: Construct, id: string, props?: cdk.StackProps) {
    super(scope, id, props);

    // The code that defines your stack goes here
    const functionName = `poker-function`;
    
　　// S3 bucket
    const bucket = new s3.Bucket(this, 'PokerBucket', {
      autoDeleteObjects: true,
      removalPolicy: cdk.RemovalPolicy.DESTROY,
    });
    
    // Print the bucket name for KVS S3 delivery configuration
    new cdk.CfnOutput(this, 'BucketNameOutput', {
      description: 'The name of the S3 bucket',
      value: bucket.bucketName,
    });

    // IAM role
    const lambdaRole = new iam.Role(this, 'LambdaExecutionRole', {
      assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
    });
    
    bucket.grantRead(lambdaRole);
    
    // Policy doc
    const lambdaPolicy = new iam.Policy(this, 'LambdaPolicy', {
      policyName: 'LambdaExecutionPolicy',
      statements: [
        new iam.PolicyStatement({
          effect: iam.Effect.ALLOW,
          actions: ['logs:CreateLogGroup'],
          resources: [`arn:aws:logs:${this.region}:${this.account}:*`],
        }),
        new iam.PolicyStatement({
          effect: iam.Effect.ALLOW,
          actions: ['logs:CreateLogStream', 'logs:PutLogEvents'],
          resources: [`arn:aws:logs:${this.region}:${this.account}:log-group:/aws/lambda/${functionName}:*`],
        }),
        new iam.PolicyStatement({
          effect: iam.Effect.ALLOW,
          actions: ['bedrock:InvokeModel'],
          resources: [`arn:aws:bedrock:${this.region}::foundation-model/*`], 
        }),
        new iam.PolicyStatement({
          effect: iam.Effect.ALLOW,
          actions: ['iot:Publish'],
          resources: [`arn:aws:iot:${this.region}:${this.account}:topic/poker-advice`], 
        }),
      ],
    });

    // Attach the policy doc
    lambdaRole.attachInlinePolicy(lambdaPolicy);    
    
    // Lambda function
    const lambdaFunction = new lambda.Function(this, 'PokerLambda', {
      code: lambda.Code.fromAsset('./lambda/'), 
      handler: 'lambda_function.lambda_handler', 
      runtime: lambda.Runtime.PYTHON_3_12, 
      role: lambdaRole,
      functionName: functionName,
      environment: {
        BUCKET_NAME: bucket.bucketName
      },
      timeout: cdk.Duration.seconds(9)
    });
    
    // S3 trigger
    bucket.addEventNotification(s3.EventType.OBJECT_CREATED, new s3n.LambdaDestination(lambdaFunction));
  }
}

