/**
 * チャットページコンポーネント
 * リアルタイムチャット機能を提供するメインページ
 */
'use client';

// 必要なReactフックとコンポーネントのインポート
import { useState, useEffect, useRef } from 'react';
// AWS Amplifyの機能をインポート
import { events } from 'aws-amplify/data';
import { Amplify } from 'aws-amplify';
// Next.jsのルーティング機能
import { useRouter } from 'next/navigation';
// カスタムコンポーネント
import MessageBubble from '../components/MessageBubble';
import MessageInput from '../components/MessageInput';
// 型定義
import { Message } from '../types/message';
// Amplify設定
import { AMPLIFY_CONFIG } from '../config/amplify';

// Amplify設定を初期化
Amplify.configure(AMPLIFY_CONFIG);

export default function ChatPage() {
  // ステート変数の定義
  const [username, setUsername] = useState(''); // ユーザー名
  const [liveId, setLiveId] = useState('');    // チャットルームID
  const [messages, setMessages] = useState<Message[]>([]); // メッセージリスト
  const [isSubmitting, setIsSubmitting] = useState(false); // 送信中フラグ
  const router = useRouter(); // ルーター

  // メッセージ一覧の最下部への参照（自動スクロール用）
  const messagesEndRef = useRef<HTMLDivElement>(null);

  /**
   * 初期化とチャットチャンネル接続のためのエフェクト
   * コンポーネントマウント時に実行される
   */
  useEffect(() => {
    // ローカルストレージからユーザー情報を取得
    const storedUsername = localStorage.getItem('chatUsername');
    const storedLiveId = localStorage.getItem('chatLiveId');

    // ユーザー情報がない場合は参加ページにリダイレクト
    if (!storedUsername || !storedLiveId) {
      router.push('/join');
      return;
    }

    // ステート変数を設定
    setUsername(storedUsername);
    setLiveId(storedLiveId);

    // チャットチャンネルに接続
    const connectionPromise = events.connect(`/chat/${storedLiveId}`);

    // 接続成功時の処理
    connectionPromise.then((channel) => {
      channel.subscribe({
        next: handleIncomingMessage, // メッセージ受信時のハンドラ
        error: (err) => alert('Subscription error: ' + err), // エラー処理
      });
    }).catch(error => {
      console.log('Error connecting to channel: ' + error);
      alert('Error connecting to channel');
    });

    // クリーンアップ関数（コンポーネントアンマウント時に実行）
    return () => {
      connectionPromise.then((channel) => channel?.close());
    };
  }, [router]);

  /**
   * メッセージリストが更新されたら自動スクロールするエフェクト
   */
  useEffect(() => {
    scrollToBottom();
  }, [messages]);

  /**
   * 受信したメッセージを処理するハンドラ
   * @param data サーバーから受信したメッセージデータ
   */
  const handleIncomingMessage = (data: any) => {
    console.log('Incoming message:', data);

    try {
      // メッセージデータを抽出
      const messageId = data.event?.messageId;
      const username = data.event?.username;
      const message = data.event?.message;
      const timestamp = data.event?.timestamp;
      const isRejected = data.event?.isRejected;

      // メッセージオブジェクトを作成
      const processedMessage: Message = {
        id: messageId,
        username: username,
        message: message,
        timestamp: timestamp,
        isRejected: isRejected
      };

      // メッセージリストを更新
      setMessages(prevMessages => [...prevMessages, processedMessage]);
      scrollToBottom();
    } catch (error) {
      console.error('Error processing message:', error);
    }
  };

  /**
   * メッセージ一覧の最下部にスクロールする関数
   * 少し遅延を入れることで、DOMの更新後に確実にスクロールされるようにする
   */
  const scrollToBottom = () => {
    setTimeout(() => {
      messagesEndRef.current?.scrollIntoView({ behavior: 'smooth', block: 'end' });
    }, 100);
  };

  /**
   * メッセージを送信する関数
   * @param message 送信するメッセージ内容
   */
  const sendMessage = async (message: string) => {
    // 空メッセージや送信中の場合は処理しない
    if (!message.trim() || isSubmitting) return;

    // 送信中フラグをセット
    setIsSubmitting(true);

    try {
      // 送信するメッセージデータを作成
      const messageData = [
        {
          liveId: liveId,
          message: message,
          username: username
        }
      ];
      scrollToBottom();
      // メッセージをサーバーに送信
      await events.post(`/chat/${liveId}`, messageData);
    } catch (error) {
      console.error('Error sending message:', error);
    } finally {
      // 送信中フラグを解除
      setIsSubmitting(false);
    }
  };

  /**
   * UIレンダリング
   */
  return (
    <div className="min-h-screen bg-gray-100 flex flex-col">
      {/* ヘッダー部分 */}
      <div className="bg-white shadow-sm border-b p-4">
        <h1 className="text-xl font-bold text-gray-800">
          Live Chat - {username} (Live ID: {liveId})
        </h1>
      </div>

      {/* チャット本体部分 */}
      <div className="flex-1 overflow-hidden flex flex-col">
        {/* メッセージ表示エリア */}
        <div className="flex-1 overflow-y-auto p-4 flex flex-col">
          <div className="flex-grow"></div>
          <div className="space-y-2">
            {/* メッセージ一覧をマッピング */}
            {messages.map((message, index) => (
              <MessageBubble key={message.id || index} message={message} />
            ))}
          </div>
          {/* 自動スクロール用の参照ポイント */}
          <div ref={messagesEndRef} />
        </div>
        {/* メッセージ入力コンポーネント */}
        <MessageInput onSendMessage={sendMessage} isSubmitting={isSubmitting} />
      </div>
    </div>
  );
}